<?php

namespace App\Http\Livewire;

use App\Models\Setting;
use App\Rules\ValidPurchaseCode;
use App\Services\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Livewire\Component;

class Settings extends Component {

	public $settings;

	public function mount(): void {
		$rawValues = Setting::whereIn( 'group', [ 'ui', 'envato', 'api', 'logging', 'productData' ] )->get()->toArray();
		$settings  = [
			"ui"          => [
				"brand"         => "Keevault",
				"itemsPerTable" => "10",
				"daysPerGraph"  => "7",
				"dateFormat"    => "F j, Y",
				"timeFormat"    => "H:i",
			],
			"envato"      => [
				"personalToken" => "",
				"createProduct" => 1
			],
			"api"         => [
				"passphrase"         => "",
				"privateKey"         => "",
				"algorithm"          => "",
				"blockedIPs"         => "",
				"blockedIdentifiers" => "",
				"acceptedUseragent"  => "",
			],
			"logging"     => [
				'dashboardActivity'   => 1,
				'apiCalls'            => 1,
				'failedApiCalls'      => 1,
				'blocklistedApiCalls' => 1,
			],
			'productData' => [
				'purchaseCode' => ""
			]
		];

		foreach ( $rawValues as $row ) {
			$settings[ $row['group'] ][ $row['name'] ] = $row['value'];
		}

		$settings['envato']['createProduct']        = (bool) $settings['envato']['createProduct'];
		$settings['logging']['dashboardActivity']   = (bool) $settings['logging']['dashboardActivity'];
		$settings['logging']['apiCalls']            = (bool) $settings['logging']['apiCalls'];
		$settings['logging']['failedApiCalls']      = (bool) $settings['logging']['failedApiCalls'];
		$settings['logging']['blocklistedApiCalls'] = (bool) $settings['logging']['blocklistedApiCalls'];

		$this->settings = $settings;
	}

	public function userInterfaceRules(): array {
		return [
			'settings.ui.brand'         => 'required|string',
			'settings.ui.itemsPerTable' => 'required|numeric|min:10',
			'settings.ui.daysPerGraph'  => [ 'required', 'numeric', Rule::in( [ 7, 15, 30 ] ) ],
			'settings.ui.dateFormat'    => 'required',
			'settings.ui.timeFormat'    => 'required'
		];
	}

	public function apiRules(): array {
		return [
			'settings.api.passphrase'         => 'string',
			'settings.api.privateKey'         => 'string',
			'settings.api.algorithm'          => [ 'integer', Rule::in( array( 1, 2, 3, 6, 7, 8, 9, 10 ) ) ],
			'settings.api.blockedIPs'         => 'string',
			'settings.api.blockedIdentifiers' => 'string',
			'settings.api.acceptedUseragent'  => 'string',
		];
	}

	public function envatoRules(): array {
		return [
			'settings.envato.personalToken' => 'string',
		];
	}

	public function loggingRules(): array {
		return [
			'settings.logging.dashboardActivity' => 'boolean',
			'settings.logging.apiCalls'          => 'boolean',
		];
	}

	public function keevaultRules(): array {
		return [
			'settings.productData.purchaseCode' => [ 'required', new ValidPurchaseCode() ],
		];
	}

	public function getMessages(): array {
		return [
			'settings.ui.brand'                  => __( 'The brand field is required.' ),
			'settings.ui.itemsPerTable.required' => __( 'The item per table field is required.' ),
			'settings.ui.itemsPerTable.min'      => __( 'The item per table field must be 10 or more.' ),
			'settings.ui.dateFormat'             => __( 'The date format field is required.' ),
			'settings.ui.timeFormat'             => __( 'The time format field is required.' ),
			'settings.productData.purchaseCode.required'     => __( 'Please enter a valid purchase code.' ),
		];
	}

	public function userInterfaceOptionUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->userInterfaceRules() );
	}

	public function apiOptionUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->apiRules() );
	}

	public function envatoOptionUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->envatoRules() );
	}

	public function loggingOptionUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->loggingRules() );
	}

	public function keevaultOptionUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->keevaultRules() );
	}

	public function saveUserInterfaceSettings(): void {
		if ( \Config::get( 'app.is_demo', false ) ) {
			$this->dispatchBrowserEvent( 'show-toast', [
				'toastType'    => 'danger',
				'toastMessage' => __( 'This action is disabled in the demo!' )
			] );

			return;
		}

		$this->validate( $this->userInterfaceRules() );

		Log::add( '150', Auth::id(), request()->ip() );

		Setting::upsert( [ 'group' => 'ui', 'name' => 'brand', 'value' => $this->settings['ui']['brand'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'ui', 'name' => 'itemsPerTable', 'value' => $this->settings['ui']['itemsPerTable'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'ui', 'name' => 'daysPerGraph', 'value' => $this->settings['ui']['daysPerGraph'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'ui', 'name' => 'dateFormat', 'value' => $this->settings['ui']['dateFormat'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'ui', 'name' => 'timeFormat', 'value' => $this->settings['ui']['timeFormat'] ], [ 'group', 'name' ], [ 'value' ] );

		$this->dispatchBrowserEvent( 'show-toast', [
			'toastType'    => 'success',
			'toastMessage' => __( 'User Interface settings saved!' )
		] );
	}

	public function saveApiSettings(): void {
		if ( \Config::get( 'app.is_demo', false ) ) {
			$this->dispatchBrowserEvent( 'show-toast', [
				'toastType'    => 'danger',
				'toastMessage' => __( 'This action is disabled in the demo!' )
			] );

			return;
		}

		$this->validate( $this->apiRules() );

		Log::add( '151', Auth::id(), request()->ip() );

		Setting::upsert( [ 'group' => 'api', 'name' => 'passphrase', 'value' => $this->settings['api']['passphrase'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'api', 'name' => 'privateKey', 'value' => $this->settings['api']['privateKey'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'api', 'name' => 'algorithm', 'value' => $this->settings['api']['algorithm'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'api', 'name' => 'blockedIPs', 'value' => $this->settings['api']['blockedIPs'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'api', 'name' => 'blockedIdentifiers', 'value' => $this->settings['api']['blockedIdentifiers'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'api', 'name' => 'acceptedUseragent', 'value' => $this->settings['api']['acceptedUseragent'] ], [ 'group', 'name' ], [ 'value' ] );

		$this->dispatchBrowserEvent( 'show-toast', [
			'toastType'    => 'success',
			'toastMessage' => __( 'API settings saved!' )
		] );
	}

	public function saveEnvatoSettings(): void {
		if ( \Config::get( 'app.is_demo', false ) ) {
			$this->dispatchBrowserEvent( 'show-toast', [
				'toastType'    => 'danger',
				'toastMessage' => __( 'This action is disabled in the demo!' )
			] );

			return;
		}

		$this->validate( $this->envatoRules() );

		Log::add( '152', Auth::id(), request()->ip() );

		Setting::upsert( [ 'group' => 'envato', 'name' => 'personalToken', 'value' => $this->settings['envato']['personalToken'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'envato', 'name' => 'createProduct', 'value' => $this->settings['envato']['createProduct'] ], [ 'group', 'name' ], [ 'value' ] );

		$this->dispatchBrowserEvent( 'show-toast', [
			'toastType'    => 'success',
			'toastMessage' => __( 'Envato settings saved!' )
		] );
	}

	public function saveLoggingSettings(): void {
		if ( \Config::get( 'app.is_demo', false ) ) {
			$this->dispatchBrowserEvent( 'show-toast', [
				'toastType'    => 'danger',
				'toastMessage' => __( 'This action is disabled in the demo!' )
			] );

			return;
		}

		$this->validate( $this->envatoRules() );

		Log::add( '152', Auth::id(), request()->ip() );

		Setting::upsert( [ 'group' => 'logging', 'name' => 'dashboardActivity', 'value' => $this->settings['logging']['dashboardActivity'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'logging', 'name' => 'apiCalls', 'value' => $this->settings['logging']['apiCalls'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'logging', 'name' => 'failedApiCalls', 'value' => $this->settings['logging']['failedApiCalls'] ], [ 'group', 'name' ], [ 'value' ] );
		Setting::upsert( [ 'group' => 'logging', 'name' => 'blocklistedApiCalls', 'value' => $this->settings['logging']['blocklistedApiCalls'] ], [ 'group', 'name' ], [ 'value' ] );

		$this->dispatchBrowserEvent( 'show-toast', [
			'toastType'    => 'success',
			'toastMessage' => __( 'Logging settings saved!' )
		] );
	}

	public function saveKeevaultSettings(): void {
		if ( \Config::get( 'app.is_demo', false ) ) {
			$this->dispatchBrowserEvent( 'show-toast', [
				'toastType'    => 'danger',
				'toastMessage' => __( 'This action is disabled in the demo!' )
			] );

			return;
		}

		$this->validate( $this->keevaultRules() );

		Log::add( '152', Auth::id(), request()->ip() );

		Setting::upsert( [ 'group' => 'productData', 'name' => 'purchaseCode', 'value' => $this->settings['productData']['purchaseCode'] ], [ 'group', 'name' ], [ 'value' ] );

		$this->dispatchBrowserEvent( 'show-toast', [
			'toastType'    => 'success',
			'toastMessage' => __( 'Keevault settings saved!' )
		] );
	}

	public function render() {
		return view( 'livewire.settings.settings' );
	}
}
